<?php
/**
 * WebDAV服务器入口文件
 * 软件介绍页面及登录跳板
 */

// 开启会话
session_start();

// 检查是否已安装
if (!file_exists(__DIR__ . '/install.lock')) {
    header('Location: install.php');
    exit;
}

// 加载配置
$config = require 'config.php';

// 连接数据库
try {
    $dbPath = __DIR__ . '/' . $config['db_file'];
    $pdo = new PDO('sqlite:' . $dbPath);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die('数据库连接失败：' . $e->getMessage());
}

// 检查是否已登录
$isLoggedIn = isset($_SESSION['user_id']);

// 处理登录
$loginError = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'login') {
    $username = isset($_POST['username']) ? trim($_POST['username']) : '';
    $password = isset($_POST['password']) ? trim($_POST['password']) : '';
    
    if (!empty($username) && !empty($password)) {
        $stmt = $pdo->prepare('SELECT * FROM users WHERE username = ? LIMIT 1');
        $stmt->execute([$username]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($user && password_verify($password, $user['password'])) {
            // 登录成功
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['is_admin'] = $user['is_admin'];
            
            // 重定向到当前页面（刷新）
            header('Location: index.php');
            exit;
        } else {
            $loginError = '用户名或密码错误';
        }
    } else {
        $loginError = '请输入用户名和密码';
    }
}

// 处理退出登录 - 使用统一登出页面
if (isset($_GET['action']) && $_GET['action'] === 'logout') {
    header('Location: logout.php');
    exit;
}

// 获取WebDAV访问URL
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
$host = $_SERVER['HTTP_HOST'];
$webdavUrl = $protocol . '://' . $host . rtrim(dirname($_SERVER['PHP_SELF']), '/') . '/webdav.php';
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>WebDAV服务器</title>
    <style>
        body {
            font-family: Arial, sans-serif;
            line-height: 1.6;
            margin: 0;
            padding: 20px;
            background-color: #f5f5f5;
            color: #333;
        }
        .container {
            max-width: 1000px;
            margin: 0 auto;
            background-color: #fff;
            padding: 20px;
            border-radius: 5px;
            box-shadow: 0 0 10px rgba(0,0,0,0.1);
        }
        h1 {
            color: #2c3e50;
            border-bottom: 1px solid #eee;
            padding-bottom: 10px;
        }
        h2 {
            color: #3498db;
            margin-top: 20px;
        }
        .error {
            color: #e74c3c;
            background-color: #fadbd8;
            padding: 10px;
            border-radius: 3px;
            margin-bottom: 20px;
        }
        .success {
            color: #27ae60;
            background-color: #d4efdf;
            padding: 10px;
            border-radius: 3px;
            margin-bottom: 20px;
        }
        .login-form {
            background-color: #f8f9fa;
            padding: 20px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        label {
            display: block;
            margin-bottom: 5px;
            font-weight: bold;
        }
        input[type="text"],
        input[type="password"] {
            width: 100%;
            padding: 8px;
            margin-bottom: 15px;
            border: 1px solid #ddd;
            border-radius: 3px;
            box-sizing: border-box;
        }
        button {
            background-color: #3498db;
            color: white;
            border: none;
            padding: 10px 15px;
            border-radius: 3px;
            cursor: pointer;
        }
        button:hover {
            background-color: #2980b9;
        }
        .feature-list {
            list-style-type: none;
            padding: 0;
        }
        .feature-list li {
            margin-bottom: 10px;
            padding-left: 20px;
            position: relative;
        }
        .feature-list li:before {
            content: "✓";
            color: #27ae60;
            position: absolute;
            left: 0;
        }
        .user-info {
            background-color: #eaf2f8;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        .action-links {
            margin-top: 20px;
        }
        .action-links a {
            display: inline-block;
            margin-right: 15px;
            text-decoration: none;
            color: #3498db;
        }
        .action-links a:hover {
            text-decoration: underline;
        }
        .webdav-url {
            background-color: #f0f0f0;
            padding: 10px;
            border-radius: 3px;
            font-family: monospace;
            word-break: break-all;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>WebDAV服务器</h1>
        
        <?php if ($isLoggedIn): ?>
            <div class="user-info">
                <p>欢迎回来，<strong><?php echo htmlspecialchars($_SESSION['username']); ?></strong>！</p>
                <div class="action-links">
                    <?php if ($_SESSION['is_admin'] == 1): ?>
                        <a href="admin.php">管理后台</a>
                    <?php endif; ?>
                    <a href="index.php?action=logout">退出登录</a>
                </div>
            </div>
            
            <h2>WebDAV访问信息</h2>
            <p>您可以使用以下格式的URL访问WebDAV服务：</p>
            <div class="webdav-url">
                <span id="webdav-url-text"><?php echo htmlspecialchars($webdavUrl); ?></span>
                <button type="button" onclick="copyWebdavUrl()" style="margin-left: 10px; padding: 2px 8px; font-size: 12px; cursor: pointer;">复制</button>
            </div>
            
            <h2>连接说明</h2>
            <p>使用您的用户名和密码连接到WebDAV服务器。以下是在不同系统上连接的方法：</p>
            
            <h3>Windows</h3>
            <ol>
                <li>打开文件资源管理器</li>
                <li>右键点击"此电脑"，选择"映射网络驱动器"</li>
                <li>选择一个驱动器盘符</li>
                <li>在"文件夹"框中输入WebDAV URL</li>
                <li>勾选"使用其他凭据连接"</li>
                <li>点击"完成"并输入您的用户名和密码</li>
            </ol>
            
            <h3>macOS</h3>
            <ol>
                <li>在Finder中，点击"前往"菜单，选择"连接服务器"</li>
                <li>在"服务器地址"框中输入WebDAV URL</li>
                <li>点击"连接"并输入您的用户名和密码</li>
            </ol>
            
            <h3>Linux</h3>
            <ol>
                <li>使用davfs2挂载WebDAV目录：<br><code>sudo mount -t davfs <?php echo htmlspecialchars($webdavUrl); ?> /mnt/webdav</code></li>
                <li>输入您的用户名和密码</li>
            </ol>
            
        <?php else: ?>
            <div class="login-form">
                <h2>用户登录</h2>
                <?php if (!empty($loginError)): ?>
                    <div class="error"><?php echo htmlspecialchars($loginError); ?></div>
                <?php endif; ?>
                
                <form method="post" action="">
                    <div>
                        <label for="username">用户名：</label>
                        <input type="text" id="username" name="username" required>
                    </div>
                    <div>
                        <label for="password">密码：</label>
                        <input type="password" id="password" name="password" required>
                    </div>
                    <input type="hidden" name="action" value="login">
                    <button type="submit">登录</button>
                </form>
            </div>
            
            <h2>关于WebDAV服务器</h2>
            <p>这是一个基于PHP的WebDAV服务器，支持多用户管理和文件访问控制。</p>
            
            <h3>主要功能</h3>
            <ul class="feature-list">
                <li>基于PHP的轻量级WebDAV服务器</li>
                <li>支持多用户管理</li>
                <li>基于SQLite的数据库存储</li>
                <li>兼容Windows、macOS和Linux客户端</li>
                <li>支持文件上传、下载、移动和删除</li>
                <li>支持目录创建和管理</li>
                <li>简单易用的管理界面</li>
            </ul>
        <?php endif; ?>
    </div>
    
    <script>
        function copyWebdavUrl() {
            // 获取干净的URL（去除前后空格）
            var url = document.getElementById('webdav-url-text').textContent.trim();
            
            // 创建toast提示元素
            function showToast(message, duration = 2000) {
                // 如果已存在toast，先移除
                const existingToast = document.getElementById('copy-toast');
                if (existingToast) {
                    existingToast.remove();
                }
                
                const toast = document.createElement('div');
                toast.id = 'copy-toast';
                toast.style.cssText = `
                    position: fixed;
                    top: 20px;
                    right: 20px;
                    background-color: #27ae60;
                    color: white;
                    padding: 12px 24px;
                    border-radius: 4px;
                    box-shadow: 0 2px 8px rgba(0,0,0,0.2);
                    z-index: 10000;
                    font-size: 14px;
                    animation: slideIn 0.3s ease-out;
                `;
                toast.textContent = message;
                
                // 添加动画样式
                const style = document.createElement('style');
                style.textContent = `
                    @keyframes slideIn {
                        from { transform: translateX(100%); opacity: 0; }
                        to { transform: translateX(0); opacity: 1; }
                    }
                    @keyframes slideOut {
                        from { transform: translateX(0); opacity: 1; }
                        to { transform: translateX(100%); opacity: 0; }
                    }
                `;
                document.head.appendChild(style);
                
                document.body.appendChild(toast);
                
                // 2秒后移除
                setTimeout(() => {
                    toast.style.animation = 'slideOut 0.3s ease-out';
                    setTimeout(() => {
                        toast.remove();
                        style.remove();
                    }, 300);
                }, duration);
            }
            
            if (navigator.clipboard) {
                navigator.clipboard.writeText(url).then(function() {
                    showToast('复制成功');
                }).catch(function(err) {
                    console.error('复制失败:', err);
                    showToast('复制失败，请重试');
                });
            } else {
                // 使用备用方法
                var textArea = document.createElement('textarea');
                textArea.value = url;
                textArea.style.position = 'fixed';
                textArea.style.left = '-9999px';
                textArea.style.top = '-9999px';
                document.body.appendChild(textArea);
                textArea.focus();
                textArea.select();
                
                try {
                    var successful = document.execCommand('copy');
                    if (successful) {
                        showToast('复制成功');
                    } else {
                        showToast('复制失败，请手动复制');
                    }
                } catch (err) {
                    console.error('复制失败:', err);
                    showToast('复制失败，请手动复制');
                }
                
                document.body.removeChild(textArea);
            }
        }
    </script>
</body>
</html>