<?php
/**
 * 纯PHP WebDAV服务器 - 数据库集成版本
 */

// 添加会话管理
session_start();

class WebDAVServer {
    private $baseDir;
    private $realm = 'WebDAV Server';
    private $authenticated = false;
    private $pdo = null;
    private $config = [];
    private $currentUser = null;
    private $userDir = null;
    
    public function __construct($baseDir = './webdav', $users = [], $realm = 'WebDAV Server') {
        // 加载配置
        $this->config = require __DIR__ . '/config.php';
        
        // 设置基础目录
        $this->baseDir = isset($this->config['base_dir']) ? rtrim($this->config['base_dir'], '/') . '/' : rtrim($baseDir, '/') . '/';
        $this->realm = $realm;
        
        // 确保基础目录存在
        if (!is_dir($this->baseDir)) {
            mkdir($this->baseDir, 0755, true);
        }
        
        // 连接数据库
        try {
            if (isset($this->config['db_file'])) {
                $dbPath = __DIR__ . '/' . $this->config['db_file'];
                if (file_exists($dbPath)) {
                    $this->pdo = new PDO('sqlite:' . $dbPath);
                    $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
                }
            }
        } catch (PDOException $e) {
            // 数据库连接失败，继续使用配置文件中的用户
        }
        
        // 初始化当前用户目录
        $this->userDir = null;
    }
    
    public function handle() {
        try {
            // 处理登出请求 - 现在重定向到统一登出页面
            if (isset($_GET['logout'])) {
                header('Location: logout.php');
                exit;
            }
            
            // 处理身份验证 - 修复版本
            if (!$this->authenticateFixed()) {
                $this->sendUnauthorized();
                return;
            }
            
            $method = $_SERVER['REQUEST_METHOD'];
            $path = $this->getPath();
            
            switch ($method) {
                case 'OPTIONS':
                    $this->handleOptions();
                    break;
                case 'GET':
                    $this->handleGet($path);
                    break;
                case 'HEAD':
                    $this->handleHead($path);
                    break;
                case 'PUT':
                    $this->handlePut($path);
                    break;
                case 'DELETE':
                    $this->handleDelete($path);
                    break;
                case 'MKCOL':
                    $this->handleMkcol($path);
                    break;
                case 'COPY':
                    $this->handleCopy($path);
                    break;
                case 'MOVE':
                    $this->handleMove($path);
                    break;
                case 'PROPFIND':
                    $this->handlePropfind($path);
                    break;
                case 'PROPPATCH':
                    $this->handleProppatch($path);
                    break;
                default:
                    $this->sendResponse(405, 'Method Not Allowed');
            }
        } catch (Exception $e) {
            $this->sendResponse(500, 'Internal Server Error: ' . $e->getMessage());
        }
    }
    
    private function authenticateFixed() {
        // 如果没有安装，允许匿名访问
        if (!file_exists(__DIR__ . '/install.lock')) {
            return true;
        }
        
        // 获取认证信息
        $username = null;
        $password = null;
        
        // 方法1: 标准PHP_AUTH
        if (isset($_SERVER['PHP_AUTH_USER']) && isset($_SERVER['PHP_AUTH_PW'])) {
            $username = $_SERVER['PHP_AUTH_USER'];
            $password = $_SERVER['PHP_AUTH_PW'];
        }
        // 方法2: HTTP_AUTHORIZATION头
        elseif (isset($_SERVER['HTTP_AUTHORIZATION'])) {
            $auth = $_SERVER['HTTP_AUTHORIZATION'];
            if (preg_match('/Basic\s+(.*)$/i', $auth, $matches)) {
                list($username, $password) = explode(':', base64_decode($matches[1]));
            }
        }
        // 方法3: REDIRECT_HTTP_AUTHORIZATION头
        elseif (isset($_SERVER['REDIRECT_HTTP_AUTHORIZATION'])) {
            $auth = $_SERVER['REDIRECT_HTTP_AUTHORIZATION'];
            if (preg_match('/Basic\s+(.*)$/i', $auth, $matches)) {
                list($username, $password) = explode(':', base64_decode($matches[1]));
            }
        }
        
        if ($username && $password) {
            // 优先使用数据库验证
            if ($this->pdo) {
                try {
                    $stmt = $this->pdo->prepare('SELECT * FROM users WHERE username = ? LIMIT 1');
                    $stmt->execute([$username]);
                    $user = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($user && password_verify($password, $user['password'])) {
                        $this->currentUser = $user;
                        $this->userDir = $user['access_dir'] ?: $username;
                        $this->authenticated = true;
                        return true;
                    }
                } catch (Exception $e) {
                    // 数据库验证失败，尝试配置文件验证
                }
            }
            
            // 回退到配置文件验证
            if (isset($this->config['users'][$username]) && $this->config['users'][$username] === $password) {
                $this->currentUser = ['username' => $username];
                $this->userDir = $username;
                $this->authenticated = true;
                return true;
            }
        }
        
        return false;
    }
    
    private function sendUnauthorized() {
        header('WWW-Authenticate: Basic realm="' . $this->realm . '"');
        header('HTTP/1.1 401 Unauthorized');
        echo '<!DOCTYPE html><html><head><title>401 Unauthorized</title></head><body><h1>401 Unauthorized</h1><p>请输入正确的用户名和密码</p></body></html>';
        exit;
    }
    

    
    // 其余方法保持不变...
    private function getPath() {
        $path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
        $path = urldecode($path);
        $path = ltrim($path, '/');
        
        // 获取当前脚本的路径
        $scriptPath = $_SERVER['SCRIPT_NAME'];
        $scriptDir = dirname($scriptPath);
        $scriptName = basename($scriptPath);
        
        // 构建webdav.php的完整路径
        $webdavPath = trim($scriptDir . '/' . $scriptName, '/');
        
        // 如果请求的是webdav.php本身，返回空字符串以访问根目录
        if ($path === $webdavPath || $path === $scriptName || $path === '') {
            return '';
        }
        
        // 如果路径以webdav.php/开头，去掉前缀
        $prefix = $webdavPath . '/';
        if (strpos($path, $prefix) === 0) {
            return substr($path, strlen($prefix));
        }
        
        // 检查是否直接访问webdav.php
        if (strpos($path, $scriptName . '/') === 0) {
            return substr($path, strlen($scriptName) + 1);
        }
        
        // 检查是否部署在根目录
        if ($scriptDir === '/') {
            return $path;
        }
        
        return $path;
    }
    
    private function getRealPath($path) {
        // 如果没有安装，使用基础目录
        if (!file_exists(__DIR__ . '/install.lock')) {
            return $this->baseDir . ltrim($path, '/');
        }
        
        // 检查当前用户是否为管理员
        $isAdmin = false;
        if ($this->currentUser && isset($this->currentUser['is_admin'])) {
            $isAdmin = (bool)$this->currentUser['is_admin'];
        }
        
        // 管理员用户不受目录限制，可以访问所有路径
        if ($isAdmin) {
            return $this->baseDir . ltrim($path, '/');
        }
        
        // 普通用户受目录限制
        if (!$this->userDir) {
            return $this->baseDir . ltrim($path, '/');
        }
        
        // 构建用户限制目录路径
        $userBasePath = $this->baseDir . $this->userDir;
        
        // 确保用户目录存在
        if (!is_dir($userBasePath)) {
            mkdir($userBasePath, 0755, true);
        }
        
        // 限制用户只能访问其指定目录
        $relativePath = ltrim($path, '/');
        return $userBasePath . '/' . $relativePath;
    }
    
    private function handleOptions() {
        header('Allow: OPTIONS, GET, HEAD, PUT, DELETE, MKCOL, COPY, MOVE, PROPFIND, PROPPATCH');
        header('DAV: 1, 2');
        $this->sendResponse(200, 'OK');
    }
    
    private function handleGet($path) {
        $realPath = $this->getRealPath($path);
        
        if (!file_exists($realPath)) {
            $this->sendResponse(404, 'Not Found');
            return;
        }
        
        if (is_dir($realPath)) {
            $this->sendDirectoryListing($realPath, $path);
        } else {
            $this->sendFile($realPath);
        }
    }
    
    private function handleHead($path) {
        $realPath = $this->getRealPath($path);
        
        if (!file_exists($realPath)) {
            $this->sendResponse(404, 'Not Found');
            return;
        }
        
        $this->setFileHeaders($realPath);
        $this->sendResponse(200, 'OK');
    }
    
    private function handlePut($path) {
        $realPath = $this->getRealPath($path);
        $dir = dirname($realPath);
        
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
        
        $input = fopen('php://input', 'rb');
        $output = fopen($realPath, 'wb');
        
        if ($input && $output) {
            stream_copy_to_stream($input, $output);
            fclose($input);
            fclose($output);
            $this->sendResponse(201, 'Created');
        } else {
            $this->sendResponse(500, 'Failed to write file');
        }
    }
    
    private function handleDelete($path) {
        $realPath = $this->getRealPath($path);
        
        if (!file_exists($realPath)) {
            $this->sendResponse(404, 'Not Found');
            return;
        }
        
        if (is_dir($realPath)) {
            $this->removeDirectory($realPath);
        } else {
            unlink($realPath);
        }
        
        $this->sendResponse(204, 'No Content');
    }
    
    private function handleMkcol($path) {
        $realPath = $this->getRealPath($path);
        
        if (file_exists($realPath)) {
            $this->sendResponse(405, 'Method Not Allowed');
            return;
        }
        
        if (mkdir($realPath, 0755, true)) {
            $this->sendResponse(201, 'Created');
        } else {
            $this->sendResponse(409, 'Conflict');
        }
    }
    
    private function handleCopy($path) {
        $source = $this->getRealPath($path);
        $destination = $this->getRealPath($_SERVER['HTTP_DESTINATION'] ?? '');
        
        if (!file_exists($source) || !$destination) {
            $this->sendResponse(404, 'Not Found');
            return;
        }
        
        if (file_exists($destination) && !isset($_SERVER['HTTP_OVERWRITE']) || $_SERVER['HTTP_OVERWRITE'] !== 'T') {
            $this->sendResponse(412, 'Precondition Failed');
            return;
        }
        
        if ($this->copy($source, $destination)) {
            $this->sendResponse(file_exists($destination) ? 204 : 201, file_exists($destination) ? 'No Content' : 'Created');
        } else {
            $this->sendResponse(500, 'Internal Server Error');
        }
    }
    
    private function handleMove($path) {
        $source = $this->getRealPath($path);
        $destination = $this->getRealPath($_SERVER['HTTP_DESTINATION'] ?? '');
        
        if (!file_exists($source) || !$destination) {
            $this->sendResponse(404, 'Not Found');
            return;
        }
        
        if (file_exists($destination) && !isset($_SERVER['HTTP_OVERWRITE']) || $_SERVER['HTTP_OVERWRITE'] !== 'T') {
            $this->sendResponse(412, 'Precondition Failed');
            return;
        }
        
        if (rename($source, $destination)) {
            $this->sendResponse(file_exists($destination) ? 204 : 201, file_exists($destination) ? 'No Content' : 'Created');
        } else {
            $this->sendResponse(500, 'Internal Server Error');
        }
    }
    
    private function handlePropfind($path) {
        $realPath = $this->getRealPath($path);
        $depth = $_SERVER['HTTP_DEPTH'] ?? '1';
        
        if (!file_exists($realPath)) {
            $this->sendResponse(404, 'Not Found');
            return;
        }
        
        $xml = $this->generatePropfindResponse($realPath, $path, $depth);
        
        header('Content-Type: application/xml; charset=utf-8');
        $this->sendResponse(207, 'Multi-Status', $xml);
    }
    
    private function handleProppatch($path) {
        $xml = '<?xml version="1.0" encoding="utf-8"?>
<d:multistatus xmlns:d="DAV:">
  <d:response>
    <d:href>' . htmlspecialchars($_SERVER['REQUEST_URI']) . '</d:href>
    <d:propstat>
      <d:prop/>
      <d:status>HTTP/1.1 200 OK</d:status>
    </d:propstat>
  </d:response>
</d:multistatus>';
        
        header('Content-Type: application/xml; charset=utf-8');
        $this->sendResponse(207, 'Multi-Status', $xml);
    }
    
    private function sendFile($realPath) {
        $this->setFileHeaders($realPath);
        readfile($realPath);
    }
    
    private function setFileHeaders($realPath) {
        $fileSize = filesize($realPath);
        $mimeType = $this->getMimeType($realPath);
        $lastModified = gmdate('D, d M Y H:i:s T', filemtime($realPath));
        $etag = md5_file($realPath);
        
        header('Content-Type: ' . $mimeType);
        header('Content-Length: ' . $fileSize);
        header('Last-Modified: ' . $lastModified);
        header('ETag: "' . $etag . '"');
    }
    
    private function sendDirectoryListing($realPath, $webPath) {
        $files = scandir($realPath);
        
        // 获取当前请求的脚本路径和基础URL
        $requestUri = $_SERVER['REQUEST_URI'];
        $scriptPath = $_SERVER['SCRIPT_NAME'];
        $basePath = dirname($scriptPath);
        $basePath = rtrim($basePath, '/') . '/';
        
        // 获取完整的协议和域名
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'];
        $baseUrl = $protocol . '://' . $host . $basePath;
        
        // 获取当前目录的相对路径
        $currentPath = trim($webPath, '/');
        
        // 计算返回上一级目录的路径
        $parentPath = '';
        if ($currentPath !== '') {
            $pathParts = explode('/', $currentPath);
            array_pop($pathParts);
            $parentPath = implode('/', $pathParts);
        }
        
        // 构建webdav.php的完整URL
        $webdavScript = basename($scriptPath);
        $webdavBase = $baseUrl . $webdavScript;

        $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Directory listing for ' . htmlspecialchars($webPath) . '</title>
    <style>
        /* 基础样式 */
        body { 
            font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Arial, sans-serif;
            margin: 0; 
            padding: 20px; 
            background-color: #f8f9fa;
            color: #333;
            line-height: 1.6;
        }
        
        .user-info {
            position: absolute;
            top: 20px;
            right: 20px;
            background: white;
            padding: 10px 15px;
            border-radius: 4px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            font-size: 14px;
        }
        
        .user-info span {
            color: #333;
            margin-right: 10px;
        }
        
        .logout-link {
            color: #dc3545;
            text-decoration: none;
            padding: 5px 10px;
            border: 1px solid #dc3545;
            border-radius: 3px;
            font-size: 12px;
        }
        
        .logout-link:hover {
            background: #dc3545;
            color: white;
        }
        
        @media (max-width: 768px) {
            .user-info {
                position: relative;
                top: 0;
                right: 0;
                margin-bottom: 15px;
                text-align: right;
            }
        }
        
        h1 { 
            color: #333; 
            margin-bottom: 20px;
            font-size: 24px;
            word-break: break-word;
        }
        
        .navigation { 
            margin-bottom: 15px; 
            display: flex;
            flex-wrap: wrap;
            gap: 6px;
        }
        
        .navigation a { 
            padding: 4px 8px; 
            background-color: #f0f0f0; 
            text-decoration: none; 
            border-radius: 3px; 
            color: #0066cc;
            font-size: 13px;
            transition: background-color 0.2s;
            border: 1px solid #ddd;
        }
        
        .navigation a:hover { 
            background-color: #e0e0e0; 
        }
        
        table { 
            width: 100%; 
            border-collapse: collapse; 
            background: white;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        th, td { 
            padding: 12px; 
            text-align: left; 
            border-bottom: 1px solid #eee; 
        }
        
        th { 
            background-color: #f8f9fa; 
            font-weight: 600;
            color: #495057;
        }
        
        tr:hover {
            background-color: #f8f9fa;
        }
        
        a { 
            text-decoration: none; 
            color: #0066cc; 
        }
        
        a:hover { 
            text-decoration: underline; 
        }
        
        .folder { 
            color: #0066cc; 
            font-weight: 600; 
        }
        
        .file { 
            color: #333; 
        }
        
        .php-file {
            color: #dc3545;
            cursor: not-allowed;
            font-style: italic;
            opacity: 0.8;
        }
        
        .php-file:hover {
            text-decoration: none;
            background-color: #fff3cd;
            padding: 2px 4px;
            border-radius: 3px;
        }
        
        /* 响应式设计 */
        @media (max-width: 768px) {
            body {
                padding: 10px;
            }
            
            h1 {
                font-size: 20px;
                margin-bottom: 15px;
            }
            
            .navigation {
                flex-direction: row;
                gap: 8px;
                flex-wrap: nowrap;
            }
            
            .navigation a {
                flex: 1;
                text-align: center;
                padding: 6px 8px;
                font-size: 13px;
                min-height: auto;
                white-space: nowrap;
            }
            
            table {
                font-size: 12px;
                border-radius: 0;
                box-shadow: none;
            }
            
            th, td {
                padding: 6px 4px;
            }
            
            /* 调整大小列字体 */
            th:nth-child(2), td:nth-child(2) {
                font-size: 10px;
                white-space: nowrap;
            }
            
            /* 调整最后修改时间列 */
            th:last-child, td:last-child {
                font-size: 10px;
                white-space: nowrap;
            }
        }
        
        @media (max-width: 480px) {
            body {
                padding: 8px;
            }
            
            h1 {
                font-size: 16px;
                margin-bottom: 12px;
            }
            
            .navigation {
                gap: 6px;
            }
            
            .navigation a {
                padding: 5px 6px;
                font-size: 12px;
            }
            
            table {
                font-size: 11px;
                min-width: 100%;
            }
            
            th, td {
                padding: 4px 3px;
            }
            
            /* 所有列都显示，但字体更小 */
            th, td {
                font-size: 10px;
            }
            
            /* 优化表格显示 */
            table {
                display: table;
                overflow-x: auto;
            }
        }
        
        @media (max-width: 360px) {
            .navigation a {
                padding: 4px 5px;
                font-size: 11px;
            }
            
            table {
                font-size: 10px;
            }
            
            th, td {
                padding: 3px 2px;
                font-size: 9px;
            }
        }
    </style>
</head>
<body>
    <div class="user-info">
        <span>👤 ' . htmlspecialchars($this->currentUser['username'] ?? '用户') . '</span>
        <a href="logout.php" class="logout-link" onclick="return confirm(\'确定要退出吗？\')">登出</a>
    </div>
    <h1>📁 ' . htmlspecialchars($webPath) . '</h1>
    
    <div class="navigation">
        <a href="' . htmlspecialchars($webdavBase) . '">🏠 根目录</a>
        ';
        
        // 添加返回上一级目录的链接（只在非根目录显示）
        if ($currentPath !== '') {
            // 如果父路径为空，则返回webdav.php根目录，否则返回上一级目录
            $parentLink = $parentPath === '' ? $webdavBase : $webdavBase . '/' . $parentPath;
            $html .= '<a href="' . htmlspecialchars($parentLink) . '">⬆️ 上一级</a>';
        }
        
        $html .= '</div>
    
    <table>
        <tr>
            <th>名称</th>
            <th>大小</th>
            <th>修改时间</th>
        </tr>';

        // 先显示目录，再显示文件
        $directories = [];
        $filesList = [];
        
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') continue;
            
            $fileRealPath = $realPath . '/' . $file;
            if (is_dir($fileRealPath)) {
                $directories[] = $file;
            } else {
                $filesList[] = $file;
            }
        }
        
        // 显示目录
        foreach ($directories as $file) {
            $fileRealPath = $realPath . '/' . $file;
            $size = '-';
            $mtime = date('Y-m-d H:i:s', filemtime($fileRealPath));
            $displayName = htmlspecialchars($file) . '/';
            
            // 构建目录URL - 使用webdav.php路径
            $url = $webdavBase . '/' . ($currentPath ? $currentPath . '/' : '') . $file . '/';
            
            $html .= '<tr>
                <td><a href="' . htmlspecialchars($url) . '" class="folder">📁 ' . $displayName . '</a></td>
                <td>' . $size . '</td>
                <td>' . $mtime . '</td>
            </tr>';
        }
        
        // 显示文件
        foreach ($filesList as $file) {
            $fileRealPath = $realPath . '/' . $file;
            $size = $this->formatBytes(filesize($fileRealPath));
            $mtime = date('Y-m-d H:i:s', filemtime($fileRealPath));
            $displayName = htmlspecialchars($file);
            
            // 检查文件扩展名
            $fileExtension = pathinfo($file, PATHINFO_EXTENSION);
            $isPhpFile = strtolower($fileExtension) === 'php';
            
            if ($isPhpFile) {
                // PHP文件显示提示信息
                $html .= '<tr>
                    <td><span class="file php-file" title="PHP文件无法在线查看与下载，请使用WebDAV客户端下载">📄 ' . $displayName . '</span></td>
                    <td>' . $size . '</td>
                    <td>' . $mtime . '</td>
                </tr>';
            } else {
                // 其他文件正常显示下载链接
                $url = $webdavBase . '/' . ($currentPath ? $currentPath . '/' : '') . $file;
                $html .= '<tr>
                    <td><a href="' . htmlspecialchars($url) . '" class="file">📄 ' . $displayName . '</a></td>
                    <td>' . $size . '</td>
                    <td>' . $mtime . '</td>
                </tr>';
            }
        }

        $html .= '</table>
</body>
</html>';
        
        header('Content-Type: text/html; charset=utf-8');
        echo $html;
    }
    
    private function generatePropfindResponse($realPath, $webPath, $depth) {
        $xml = '<?xml version="1.0" encoding="utf-8"?>
<d:multistatus xmlns:d="DAV:">
';
        
        $items = [$realPath];
        if ($depth === '1' && is_dir($realPath)) {
            $items = array_merge($items, glob($realPath . '/*'));
        }
        
        // 获取webdav.php的完整URL
        $scriptPath = $_SERVER['SCRIPT_NAME'];
        $basePath = dirname($scriptPath);
        $basePath = rtrim($basePath, '/') . '/';
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'];
        $webdavBase = $protocol . '://' . $host . $basePath . basename($scriptPath);
        
        foreach ($items as $item) {
            $itemWebPath = str_replace($this->baseDir, '', $item);
            $itemWebPath = ltrim($itemWebPath, '/');
            $href = $webdavBase . '/' . $itemWebPath;
            
            $isDir = is_dir($item);
            $size = $isDir ? 0 : filesize($item);
            $mtime = filemtime($item);
            
            $xml .= '  <d:response>
    <d:href>' . htmlspecialchars($href) . '</d:href>
    <d:propstat>
      <d:prop>
        <d:displayname>' . htmlspecialchars(basename($item)) . '</d:displayname>
        <d:getcontentlength>' . $size . '</d:getcontentlength>
        <d:getcontenttype>' . ($isDir ? 'httpd/unix-directory' : $this->getMimeType($item)) . '</d:getcontenttype>
        <d:resourcetype>' . ($isDir ? '<d:collection/>' : '') . '</d:resourcetype>
        <d:getlastmodified>' . gmdate('D, d M Y H:i:s T', $mtime) . '</d:getlastmodified>
        <d:creationdate>' . gmdate('Y-m-d\TH:i:s\Z', $mtime) . '</d:creationdate>
        <d:supportedlock>
          <d:lockentry>
            <d:lockscope><d:exclusive/></d:lockscope>
            <d:locktype><d:write/></d:locktype>
          </d:lockentry>
          <d:lockentry>
            <d:lockscope><d:shared/></d:lockscope>
            <d:locktype><d:write/></d:locktype>
          </d:lockentry>
        </d:supportedlock>
      </d:prop>
      <d:status>HTTP/1.1 200 OK</d:status>
    </d:propstat>
  </d:response>
';
        }
        
        $xml .= '</d:multistatus>';
        return $xml;
    }
    
    private function copy($source, $destination) {
        if (is_dir($source)) {
            if (!is_dir($destination)) {
                mkdir($destination, 0755, true);
            }
            
            $files = scandir($source);
            foreach ($files as $file) {
                if ($file === '.' || $file === '..') continue;
                
                $src = $source . '/' . $file;
                $dst = $destination . '/' . $file;
                
                if (is_dir($src)) {
                    $this->copy($src, $dst);
                } else {
                    copy($src, $dst);
                }
            }
            return true;
        } else {
            return copy($source, $destination);
        }
    }
    
    private function removeDirectory($dir) {
        if (!is_dir($dir)) return;
        
        $files = array_diff(scandir($dir), ['.', '..']);
        foreach ($files as $file) {
            $path = $dir . '/' . $file;
            if (is_dir($path)) {
                $this->removeDirectory($path);
            } else {
                unlink($path);
            }
        }
        return rmdir($dir);
    }
    
    private function formatBytes($bytes, $precision = 2) {
        $units = ['B', 'KB', 'MB', 'GB', 'TB'];
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }
    
    private function getMimeType($path) {
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $path);
        finfo_close($finfo);
        return $mimeType ?: 'application/octet-stream';
    }
    
    private function sendResponse($code, $message, $body = '') {
        http_response_code($code);
        if ($body) {
            echo $body;
        }
        exit;
    }
}

// 使用配置
$config = require 'config.php';
$webdav = new WebDAVServer($config['base_dir'], $config['users']);
$webdav->handle();